<?php
session_start();
if (!isset($_SESSION['student_id'])) {
    header('Location: student_login.php');
    exit;
}

// Handle language selection
if (isset($_GET['lang'])) {
    $lang = $_GET['lang'];
    if (in_array($lang, ['en', 'hi'])) {
        $_SESSION['lang'] = $lang;
    }
}
$lang = $_SESSION['lang'] ?? 'en';

// Load language strings
$lang_strings = include 'lang.php';

require_once 'connectiondb.php';



// Fetch student name
$stmt = $pdo->prepare("SELECT name FROM students WHERE id = ?");
$stmt->execute([$_SESSION['student_id']]);
$student = $stmt->fetch(PDO::FETCH_ASSOC);
$student_name = $student ? $student['name'] : 'Student';

// Fetch exams assigned to the student with attempts and allowed_attempts
$stmt = $pdo->prepare("
    SELECT e.id, e.exam_name, e.duration, se.status, se.attempts, e.allowed_attempts
    FROM exams e
    JOIN student_exams se ON e.id = se.exam_id
    WHERE se.student_id = ?
    ORDER BY e.exam_name
");
$stmt->execute([$_SESSION['student_id']]);
$exams = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Fetch questions count for each exam
$exam_ids = array_column($exams, 'id');
$questions_count = [];
if (!empty($exam_ids)) {
    $placeholders = implode(',', array_fill(0, count($exam_ids), '?'));
    $stmt = $pdo->prepare("SELECT exam_id, COUNT(*) as qcount FROM questions WHERE exam_id IN ($placeholders) GROUP BY exam_id");
    $stmt->execute($exam_ids);
    $counts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($counts as $count) {
        $questions_count[$count['exam_id']] = $count['qcount'];
    }
}
?>
<!DOCTYPE html>
<html lang="<?php echo htmlspecialchars($lang); ?>">
<head>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <title>Student Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        .exam-card:hover {
            box-shadow: 0 0 11px rgba(33,33,33,.2);
            cursor: pointer;
            transform: translateY(-3px);
            transition: all 0.3s ease;
        }
        .badge-assigned {
            background-color: #0d6efd;
        }
        .badge-completed {
            background-color: #198754;
        }
        .no-exams-illustration {
            max-width: 300px;
            margin: 20px auto;
            display: block;
        }
    </style>
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
  <div class="container-fluid">
    <a class="navbar-brand" href="#">MCQ Exam Portal</a>
    <div class="d-flex align-items-center">
      <span class="navbar-text text-white me-3"><?php echo htmlspecialchars($lang_strings[$lang]['welcome']); ?>, <?php echo htmlspecialchars($student_name); ?></span>
      <form method="get" class="me-3">
        <select name="lang" onchange="this.form.submit()" class="form-select form-select-sm" aria-label="Select language">
          <option value="en" <?php if($lang === 'en') echo 'selected'; ?>>English</option>
          <option value="hi" <?php if($lang === 'hi') echo 'selected'; ?>>हिन्दी</option>
        </select>
      </form>
      <a href="student_logout.php" class="btn btn-outline-light btn-sm"><?php echo htmlspecialchars($lang_strings[$lang]['logout']); ?></a>
    </div>
  </div>
</nav>
<div class="container my-4">
<?php
// Check if student has any approved results to show the result button
$show_result_button = false;
try {
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM student_exams WHERE student_id = ? AND status = 'completed' AND result_approved = 1");
    $stmt->execute([$_SESSION['student_id']]);
    $count = (int)$stmt->fetchColumn();
    if ($count > 0) {
        $show_result_button = true;
    }
} catch (PDOException $e) {
    // Fail silently, do not show button
    $show_result_button = false;
}
?>
    <?php if ($show_result_button): ?>
    <div class="mb-3">
        <a href="student_result.php" class="btn btn-success">View Your Results</a>
    </div>
    <?php endif; ?>
    <h2 class="mb-4"><?php echo htmlspecialchars($lang_strings[$lang]['your_exams']); ?></h2>
    <input type="text" id="searchInput" class="form-control mb-4" placeholder="<?php echo htmlspecialchars($lang_strings[$lang]['search_exams']); ?>" aria-label="Search exams" />

    <?php if (empty($exams)): ?>
        <div class="text-center">
            <img src="templates/no_exams.svg" alt="No exams" class="no-exams-illustration" />
            <p class="lead"><?php echo htmlspecialchars($lang_strings[$lang]['no_exams']); ?></p>
        </div>
    <?php else: ?>
        <div class="row" id="examsContainer">
            <?php foreach ($exams as $exam): ?>
            <div class="col-md-6 col-lg-4 mb-4 exam-card">
                <div class="card h-100 shadow-sm d-flex flex-column">
                    <div class="card-body d-flex flex-column">
                        <h5 class="card-title"><?php echo htmlspecialchars($exam['exam_name']); ?></h5>
                        <p class="card-text mb-2"><?php echo htmlspecialchars($lang_strings[$lang]['questions']); ?>: <?php echo isset($questions_count[$exam['id']]) ? $questions_count[$exam['id']] : 0; ?></p>
                        <p class="card-text mb-2">Duration: <?php echo htmlspecialchars($exam['duration']); ?> minutes</p>
                        <p class="card-text mb-2">Attempts Used: <?php echo htmlspecialchars($exam['attempts']); ?> / <?php echo htmlspecialchars($exam['allowed_attempts']); ?></p>
                        <?php if ($exam['status'] === 'assigned'): ?>
                            <?php if ($exam['attempts'] >= $exam['allowed_attempts']): ?>
                                <span class="badge bg-danger text-white mb-3">Max Attempts Reached</span>
                                <button type="button" class="btn btn-secondary mt-auto" disabled title="You have reached the maximum number of attempts. Please contact admin for approval.">Start Exam</button>
                            <?php else: ?>
                                <span class="badge badge-assigned text-white mb-3"><?php echo htmlspecialchars($lang_strings[$lang]['assigned']); ?></span>
                                <button type="button" class="btn btn-primary mt-auto start-exam-btn" data-exam-id="<?php echo $exam['id']; ?>" data-bs-toggle="tooltip" data-bs-placement="top" title="<?php echo htmlspecialchars($lang_strings[$lang]['start_exam']); ?>"><?php echo htmlspecialchars($lang_strings[$lang]['start_exam']); ?></button>
                            <?php endif; ?>
                        <?php elseif ($exam['status'] === 'completed'): ?>
                            <span class="badge badge-completed text-white mb-3"><?php echo htmlspecialchars($lang_strings[$lang]['completed']); ?></span>
                            <?php if ($exam['attempts'] < $exam['allowed_attempts']): ?>
                                <button type="button" class="btn btn-warning mt-auto start-exam-btn" data-exam-id="<?php echo $exam['id']; ?>" data-bs-toggle="tooltip" data-bs-placement="top" title="Retake Exam">Re-exam</button>
                            <?php else: ?>
                                <button type="button" class="btn btn-secondary mt-auto" disabled title="No attempts left for re-exam">Re-exam</button>
                            <?php endif; ?>
                        <?php else: ?>
                            <span class="badge bg-secondary mb-3"><?php echo htmlspecialchars($lang_strings[$lang]['unknown_status']); ?></span>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
    <?php endif; ?>
</div>
<?php include 'student_dashboard_modal_terms.php'; ?>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
<script>
    // Enable Bootstrap tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'))
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
      return new bootstrap.Tooltip(tooltipTriggerEl)
    })

    // Search filter for exams
    const searchInput = document.getElementById('searchInput');
    const examsContainer = document.getElementById('examsContainer');
    const examCards = examsContainer ? examsContainer.getElementsByClassName('exam-card') : [];

    searchInput && searchInput.addEventListener('input', function() {
        const filter = this.value.toLowerCase();
        Array.from(examCards).forEach(card => {
            const title = card.querySelector('.card-title').textContent.toLowerCase();
            if (title.includes(filter)) {
                card.style.display = '';
            } else {
                card.style.display = 'none';
            }
        });
    });

    // Removed direct start exam button click handler to allow modal event delegation to handle it
</script>
</body>
</html>
