<?php
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}
require_once 'connectiondb.php';

$exam_id = $_GET['exam_id'] ?? null;
if (!$exam_id) {
    die("Exam ID is required.");
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $exam_name = $_POST['exam_name'] ?? '';
    $exam_code = $_POST['exam_code'] ?? '';
    $duration = $_POST['duration'] ?? null;
    $exam_month = $_POST['exam_month'] ?? null;

    if (empty($exam_name) || empty($exam_code) || empty($exam_month)) {
        $error_message = "Please fill all required fields including exam month.";
    } else {
        try {
            $pdo->beginTransaction();

            // Update exam details
            $stmt = $pdo->prepare("UPDATE exams SET exam_name = ?, exam_code = ?, duration = ?, exam_month = ? WHERE id = ?");
            $stmt->execute([$exam_name, $exam_code, $duration, $exam_month, $exam_id]);

            // Check if CSV file uploaded for bulk update
            if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] === UPLOAD_ERR_OK) {
                $fileTmpPath = $_FILES['csv_file']['tmp_name'];

                // Delete existing questions and options for this exam
                $stmt = $pdo->prepare("SELECT id FROM questions WHERE exam_id = ?");
                $stmt->execute([$exam_id]);
                $question_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

                if (!empty($question_ids)) {
                    // Delete options
                    $inQuery = implode(',', array_fill(0, count($question_ids), '?'));
                    $stmt = $pdo->prepare("DELETE FROM options WHERE question_id IN ($inQuery)");
                    $stmt->execute($question_ids);

                    // Delete questions
                    $stmt = $pdo->prepare("DELETE FROM questions WHERE id IN ($inQuery)");
                    $stmt->execute($question_ids);
                }

                // Process CSV file
                if (($handle = fopen($fileTmpPath, "r")) !== false) {
                    $rowIndex = 0;
                    while (($data = fgetcsv($handle, 1000, ",")) !== false) {
                        if ($rowIndex == 0) {
                            // Skip header row
                            $rowIndex++;
                            continue;
                        }
                        $question_text = trim($data[0] ?? '');
                        $options = array_slice($data, 1, 4);
                        $correct_option = $data[5] ?? '';

                        if (empty($question_text) || count($options) < 4 || $correct_option === '') {
                            $rowIndex++;
                            continue;
                        }

                        // Adjust correct_option to zero-based index
                        $correct_option = intval($correct_option) - 1;

                        $stmt = $pdo->prepare("INSERT INTO questions (exam_id, question_text) VALUES (?, ?)");
                        $stmt->execute([$exam_id, $question_text]);
                        $question_id = $pdo->lastInsertId();

                        foreach ($options as $key => $option_text) {
                            $is_correct = ($key == $correct_option) ? 1 : 0;
                            $stmt = $pdo->prepare("INSERT INTO options (question_id, option_text, is_correct) VALUES (?, ?, ?)");
                            $stmt->execute([$question_id, $option_text, $is_correct]);
                        }
                        $rowIndex++;
                    }
                    fclose($handle);
                }
            } else {
                // Manual update of questions and options
                $questions_post = $_POST['questions'] ?? [];
                foreach ($questions_post as $qid => $qdata) {
                    // Update question text
                    $stmt = $pdo->prepare("UPDATE questions SET question_text = ? WHERE id = ?");
                    $stmt->execute([$qdata['question_text'], $qid]);

                    // Update options
                    foreach ($qdata['options'] as $oid => $odata) {
                        $is_correct = isset($qdata['correct_option']) && $qdata['correct_option'] == $oid ? 1 : 0;
                        $stmt = $pdo->prepare("UPDATE options SET option_text = ?, is_correct = ? WHERE id = ?");
                        $stmt->execute([$odata['option_text'], $is_correct, $oid]);
                    }
                }
                // Insert new questions and options
                $new_questions = $_POST['new_questions'] ?? [];
                foreach ($new_questions as $nqid => $nqdata) {
                    $stmt = $pdo->prepare("INSERT INTO questions (exam_id, question_text) VALUES (?, ?)");
                    $stmt->execute([$exam_id, $nqdata['question_text']]);
                    $new_question_id = $pdo->lastInsertId();

                    foreach ($nqdata['options'] as $index => $nopt) {
                        $is_correct = isset($nqdata['correct_option']) && $nqdata['correct_option'] == $index ? 1 : 0;
                        $stmt = $pdo->prepare("INSERT INTO options (question_id, option_text, is_correct) VALUES (?, ?, ?)");
                        $stmt->execute([$new_question_id, $nopt['option_text'], $is_correct]);
                    }
                }
            }

            $pdo->commit();
            $success_message = "Exam and questions updated successfully.";
        } catch (Exception $e) {
            $pdo->rollBack();
            $error_message = "Failed to update exam and questions: " . $e->getMessage();
        }
    }
}

// Fetch exam details
$stmt = $pdo->prepare("SELECT * FROM exams WHERE id = ?");
$stmt->execute([$exam_id]);
$exam = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$exam) {
    die("Exam not found.");
}

// Fetch questions and options
$stmt = $pdo->prepare("SELECT q.id as question_id, q.question_text, o.id as option_id, o.option_text, o.is_correct
                       FROM questions q
                       JOIN options o ON q.id = o.question_id
                       WHERE q.exam_id = ?");
$stmt->execute([$exam_id]);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

$questions = [];
foreach ($rows as $row) {
    $qid = $row['question_id'];
    if (!isset($questions[$qid])) {
        $questions[$qid] = [
            'question_text' => $row['question_text'],
            'options' => []
        ];
    }
    $questions[$qid]['options'][] = [
        'option_id' => $row['option_id'],
        'option_text' => $row['option_text'],
        'is_correct' => $row['is_correct']
    ];
}

// Fetch students who submitted this exam and their stats
$stmt = $pdo->prepare("
    SELECT s.id as student_id, s.student_id as student_code, s.name,
    COUNT(sa.id) as answers_submitted,
    SUM(CASE WHEN o.is_correct = 1 THEN 1 ELSE 0 END) as correct_answers
    FROM students s
    JOIN student_answers sa ON s.id = sa.student_id
    JOIN options o ON sa.selected_option_id = o.id
    JOIN questions q ON sa.question_id = q.id
    WHERE q.exam_id = ?
    GROUP BY s.id
");
$stmt->execute([$exam_id]);
$student_submissions = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html>
<head>
    <title>Edit Exam - <?php echo htmlspecialchars($exam['exam_name']); ?></title>
    <style>
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: #f0f0f0;
            margin: 20px;
        }
        h2, h3 {
            color: #f9fafb;
            text-shadow: 1px 1px 3px rgba(0,0,0,0.3);
        }
        .question-block {
            background-color: rgba(255, 255, 255, 0.15);
            border: 1px solid rgba(255, 255, 255, 0.3);
            border-radius: 12px;
            padding: 20px 25px;
            margin-bottom: 25px;
            box-shadow: 0 8px 20px rgba(0,0,0,0.2);
            transition: box-shadow 0.4s ease, transform 0.3s ease;
        }
        .question-block:hover {
            box-shadow: 0 12px 30px rgba(0,0,0,0.4);
            transform: translateY(-5px);
        }
        table {
            border-collapse: separate;
            border-spacing: 0 10px;
            width: 100%;
            margin-top: 25px;
            background-color: rgba(255, 255, 255, 0.15);
            border-radius: 12px;
            overflow: hidden;
            box-shadow: 0 8px 20px rgba(0,0,0,0.2);
        }
        th, td {
            border: none;
            padding: 15px 20px;
            text-align: left;
            background-color: rgba(255, 255, 255, 0.25);
            color: #f0f0f0;
            font-weight: 500;
            transition: background-color 0.3s ease;
        }
        th {
            background-color: rgba(255, 255, 255, 0.35);
            font-weight: 700;
            font-size: 16px;
        }
        tr:hover td {
            background-color: rgba(255, 255, 255, 0.45);
        }
        input[type="text"] {
            width: 90%;
            padding: 10px 12px;
            margin-bottom: 10px;
            border: none;
            border-radius: 8px;
            font-size: 15px;
            background-color: rgba(255, 255, 255, 0.3);
            color: #222;
            box-shadow: inset 0 0 8px rgba(0,0,0,0.1);
            transition: background-color 0.3s ease, box-shadow 0.3s ease;
        }
        input[type="text"]:focus {
            background-color: #fff;
            outline: none;
            box-shadow: 0 0 12px rgba(102, 126, 234, 0.8);
            color: #333;
        }
        .correct-radio {
            margin-left: 10px;
            font-weight: 600;
            color: #a3d9a5;
        }
        .option-correct input[type="text"] {
            border: 2px solid #4caf50;
            background-color: #d4edda;
            color: #2e7d32;
        }
        .option-incorrect input[type="text"] {
            border: 2px solid #e53935;
            background-color: #f8d7da;
            color: #b71c1c;
        }
        button[type="submit"], .btn-primary, .btn-secondary {
            background: linear-gradient(45deg, #667eea, #764ba2);
            color: white;
            border: none;
            padding: 14px 30px;
            font-size: 18px;
            border-radius: 10px;
            cursor: pointer;
            transition: background 0.4s ease, box-shadow 0.4s ease;
            margin-top: 15px;
            box-shadow: 0 6px 15px rgba(102, 126, 234, 0.6);
        }
        button[type="submit"]:hover, .btn-primary:hover, .btn-secondary:hover {
            background: linear-gradient(45deg, #5a67d8, #6b46c1);
            box-shadow: 0 8px 20px rgba(107, 70, 193, 0.8);
        }
        .btn-sm {
            padding: 8px 16px;
            font-size: 15px;
            border-radius: 6px;
        }
        .message {
            padding: 15px 20px;
            margin-bottom: 25px;
            border-radius: 10px;
            font-weight: 700;
            box-shadow: 0 3px 8px rgba(0,0,0,0.15);
        }
        .success {
            background-color: #4caf50;
            color: #e8f5e9;
            border: 1px solid #388e3c;
        }
        .error {
            background-color: #e53935;
            color: #ffebee;
            border: 1px solid #b71c1c;
        }
    </style>
</head>
<body>
    <h2>Edit Exam: <?php echo htmlspecialchars($exam['exam_name']); ?></h2>
    <p><strong>Total Questions: <?php echo count($questions); ?></strong></p>

    <?php if (!empty($success_message)): ?>
        <div class="message success"><?php echo htmlspecialchars($success_message); ?></div>
    <?php endif; ?>
    <?php if (!empty($error_message)): ?>
        <div class="message error"><?php echo htmlspecialchars($error_message); ?></div>
    <?php endif; ?>

<form method="POST" action="edit_exam.php?exam_id=<?php echo urlencode($exam_id); ?>" enctype="multipart/form-data">
    <h3>Exam Details</h3>
    <label>Exam Name:</label><br>
    <input type="text" name="exam_name" value="<?php echo htmlspecialchars($exam['exam_name']); ?>" required><br><br>
    <label>Exam Code:</label><br>
    <input type="text" name="exam_code" value="<?php echo htmlspecialchars($exam['exam_code']); ?>" required><br><br>
    <label>Exam Month:</label><br>
    <select name="exam_month" required class="form-select" style="max-width: 300px;">
        <option value="" disabled>Select month</option>
        <?php
        $months = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
        foreach ($months as $month) {
            $selected = ($exam['exam_month'] === $month) ? 'selected' : '';
            echo "<option value=\"$month\" $selected>$month</option>";
        }
        ?>
    </select><br><br>
    <label>Duration (minutes):</label><br>
    <input type="number" name="duration" min="1" max="600" value="<?php echo htmlspecialchars($exam['duration']); ?>" placeholder="Enter duration in minutes" class="form-control" style="max-width: 300px;"><br><br>

    <h3>Questions and Options</h3>
    <div id="questions-container">
    <?php foreach ($questions as $qid => $q): ?>
        <div class="question-block" data-question-id="<?php echo $qid; ?>">
            <p><strong>Question:</strong></p>
            <input type="text" name="questions[<?php echo $qid; ?>][question_text]" value="<?php echo htmlspecialchars($q['question_text']); ?>" required>
            <ul>
                <?php 
                $optionLabels = ['a', 'b', 'c', 'd'];
                foreach ($q['options'] as $index => $opt): ?>
                    <li class="<?php echo $opt['is_correct'] ? 'option-correct' : 'option-incorrect'; ?>">
                        <span style="font-weight:bold; margin-right:5px;"><?php echo $optionLabels[$index] ?? ''; ?>.</span>
                        <input type="text" name="questions[<?php echo $qid; ?>][options][<?php echo $opt['option_id']; ?>][option_text]" value="<?php echo htmlspecialchars($opt['option_text']); ?>" required>
                        <label class="correct-radio" for="correct_option_<?php echo $opt['option_id']; ?>" style="cursor: pointer;">
                            <input type="radio" id="correct_option_<?php echo $opt['option_id']; ?>" name="questions[<?php echo $qid; ?>][correct_option]" value="<?php echo $opt['option_id']; ?>" <?php if ($opt['is_correct']) echo 'checked'; ?>>
                            Correct
                        </label>
                    </li>
                <?php endforeach; ?>
            </ul>
            <button type="button" class="btn btn-sm btn-secondary add-option-btn">Add Option</button>
        </div>
    <?php endforeach; ?>
    </div>
    <button type="button" id="add-question-btn" class="btn btn-primary mb-3">Add New Question</button>

    <h3>Upload CSV to Update All Questions</h3>
    <input type="file" name="csv_file" accept=".csv"><br><br>
    <button type="submit">Save Changes</button>
</form>

<script>
document.addEventListener('DOMContentLoaded', function() {
    let questionCount = <?php echo count($questions); ?>;
    const questionsContainer = document.getElementById('questions-container');
    const addQuestionBtn = document.getElementById('add-question-btn');

    addQuestionBtn.addEventListener('click', function() {
        questionCount++;
        const questionBlock = document.createElement('div');
        questionBlock.className = 'question-block';
        questionBlock.dataset.questionId = 'new_' + questionCount;
        questionBlock.innerHTML = '<p><strong>Question:</strong></p>' +
            '<input type="text" name="new_questions[new_' + questionCount + '][question_text]" required>' +
            '<ul>' +
                '<li>' +
                    '<input type="text" name="new_questions[new_' + questionCount + '][options][0][option_text]" required>' +
                    '<label class="correct-radio" for="new_correct_option_' + questionCount + '_0" style="cursor: pointer;">' +
                        '<input type="radio" id="new_correct_option_' + questionCount + '_0" name="new_questions[new_' + questionCount + '][correct_option]" value="0" required>' +
                        'Correct' +
                    '</label>' +
                '</li>' +
                '<li>' +
                    '<input type="text" name="new_questions[new_' + questionCount + '][options][1][option_text]" required>' +
                    '<label class="correct-radio" for="new_correct_option_' + questionCount + '_1" style="cursor: pointer;">' +
                        '<input type="radio" id="new_correct_option_' + questionCount + '_1" name="new_questions[new_' + questionCount + '][correct_option]" value="1" required>' +
                        'Correct' +
                    '</label>' +
                '</li>' +
                '<li>' +
                    '<input type="text" name="new_questions[new_' + questionCount + '][options][2][option_text]" required>' +
                    '<label class="correct-radio" for="new_correct_option_' + questionCount + '_2" style="cursor: pointer;">' +
                        '<input type="radio" id="new_correct_option_' + questionCount + '_2" name="new_questions[new_' + questionCount + '][correct_option]" value="2" required>' +
                        'Correct' +
                    '</label>' +
                '</li>' +
                '<li>' +
                    '<input type="text" name="new_questions[new_' + questionCount + '][options][3][option_text]" required>' +
                    '<label class="correct-radio" for="new_correct_option_' + questionCount + '_3" style="cursor: pointer;">' +
                        '<input type="radio" id="new_correct_option_' + questionCount + '_3" name="new_questions[new_' + questionCount + '][correct_option]" value="3" required>' +
                        'Correct' +
                    '</label>' +
                '</li>' +
            '</ul>' +
            '<button type="button" class="btn btn-sm btn-secondary add-option-btn">Add Option</button>';
        questionsContainer.appendChild(questionBlock);
    });

    questionsContainer.addEventListener('click', function(e) {
        if (e.target.classList.contains('add-option-btn')) {
            const questionBlock = e.target.closest('.question-block');
            const ul = questionBlock.querySelector('ul');
            const options = ul.querySelectorAll('li');
            const newIndex = options.length;
            const questionId = questionBlock.dataset.questionId;
            const inputNamePrefix = questionId.startsWith('new_') ? 'new_questions' : 'questions';

            const li = document.createElement('li');
            li.innerHTML = '<input type="text" name="' + inputNamePrefix + '[' + questionId + '][options][' + newIndex + '][option_text]" required>' +
                '<label class="correct-radio">' +
                    '<input type="radio" name="' + inputNamePrefix + '[' + questionId + '][correct_option]" value="' + newIndex + '" required>' +
                    'Correct' +
                '</label>';
            ul.appendChild(li);
        }
    });
});
</script>

    <h3>Student Submissions</h3>
    <?php if (empty($student_submissions)) { ?>
        <p>No student submissions for this exam yet.</p>
    <?php } else { ?>
        <table>
            <thead>
                <tr>
                    <th>Student ID</th>
                    <th>Name</th>
                    <th>Answers Submitted</th>
                    <th>Correct Answers</th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($student_submissions as $sub): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($sub['student_code']); ?></td>
                        <td><?php echo htmlspecialchars($sub['name']); ?></td>
                        <td><?php echo $sub['answers_submitted']; ?></td>
                        <td><?php echo $sub['correct_answers']; ?></td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php } ?>

    <br>
    <a href="admin_dashboard.php">Back to Dashboard</a>
</body>
</html>
