<?php
session_start();
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header('Location: admin_login.php');
    exit;
}

if (!isset($_GET['exam_id'])) {
    die('Exam ID is required.');
}

$exam_id = intval($_GET['exam_id']);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <title>Manage Exam Questions</title>
    <meta charset="UTF-8" />
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        body {
            background: linear-gradient(135deg,rgb(16, 99, 253) 0%,rgb(255, 255, 255) 100%);
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            color: #333;
            padding: 20px;
        }
        .question-block {
            border: 1px solid #ccc;
            padding: 15px;
            margin-bottom: 20px;
            position: relative;
            border-radius: 8px;
            background-color: #fff;
        }
        .option-block {
            margin-left: 20px;
            margin-bottom: 10px;
            position: relative;
        }
        label {
            display: block;
            margin-bottom: 5px;
        }
        input[type="text"], textarea {
            width: 100%;
            padding: 8px;
            margin-bottom: 10px;
            border-radius: 5px;
            border: 1px solid #ccc;
        }
        .correct-label {
            color: green;
            font-weight: bold;
        }
        .btn {
            padding: 10px 15px;
            background-color: #007BFF;
            color: white;
            border: none;
            cursor: pointer;
            border-radius: 5px;
        }
        .btn:hover {
            background-color: #0056b3;
        }
        .remove-btn {
            position: absolute;
            top: 10px;
            right: 10px;
            background: #dc3545;
            border: none;
            color: white;
            padding: 5px 10px;
            cursor: pointer;
            border-radius: 5px;
        }
        .remove-btn:hover {
            background: #c82333;
        }
        .add-btn {
            margin-top: 10px;
        }
        .message {
            margin-bottom: 20px;
            padding: 10px;
            border-radius: 5px;
            text-align: center;
        }
        .success {
            background-color: #d4edda;
            color: #155724;
        }
        .error {
            background-color: #f8d7da;
            color: #721c24;
        }
    </style>
</head>
<body>
    <h2>Manage Questions for Exam ID: <?php echo htmlspecialchars($exam_id); ?></h2>
    <div id="message" class="message" style="display:none;"></div>
    <div id="questionsContainer"></div>
    <button id="addQuestionBtn" class="btn add-btn">Add New Question</button>
    <br /><br />
    <button id="saveChangesBtn" class="btn">Save Changes</button>

    <template id="questionTemplate">
        <div class="question-block" data-index="__INDEX__">
            <button type="button" class="remove-btn remove-question-btn" title="Remove Question">&times;</button>
            <label>Question __NUMBER__:</label>
            <textarea name="questions[__INDEX__][text]" rows="3" required></textarea>
            <input type="hidden" name="questions[__INDEX__][id]" value="0" />
            <div class="options-container">
                <div class="option-block" data-index="0">
                    <button type="button" class="remove-btn remove-option-btn" title="Remove Option">&times;</button>
                    <label>Option A:</label>
                    <input type="text" name="questions[__INDEX__][options][0][text]" required />
                    <label class="correct-label">
                        <input type="checkbox" name="questions[__INDEX__][options][0][is_correct]" value="1" />
                        Correct
                    </label>
                    <input type="hidden" name="questions[__INDEX__][options][0][id]" value="0" />
                </div>
            </div>
            <button type="button" class="btn add-option-btn add-btn">Add Option</button>
        </div>
    </template>

    <template id="optionTemplate">
        <div class="option-block" data-index="__OPT_INDEX__">
            <button type="button" class="remove-btn remove-option-btn" title="Remove Option">&times;</button>
            <label>Option __OPT_LETTER__:</label>
            <input type="text" name="questions[__Q_INDEX__][options][__OPT_INDEX__][text]" required />
            <label class="correct-label">
                <input type="checkbox" name="questions[__Q_INDEX__][options][__OPT_INDEX__][is_correct]" value="1" />
                Correct
            </label>
            <input type="hidden" name="questions[__Q_INDEX__][options][__OPT_INDEX__][id]" value="0" />
        </div>
    </template>

    <script>
        document.addEventListener('DOMContentLoaded', function () {
            const examId = <?php echo json_encode($exam_id); ?>;
            const questionsContainer = document.getElementById('questionsContainer');
            const addQuestionBtn = document.getElementById('addQuestionBtn');
            const saveChangesBtn = document.getElementById('saveChangesBtn');
            const messageDiv = document.getElementById('message');
            const questionTemplate = document.getElementById('questionTemplate').innerHTML;
            const optionTemplate = document.getElementById('optionTemplate').innerHTML;

            function showMessage(text, isSuccess) {
                messageDiv.textContent = text;
                messageDiv.className = 'message ' + (isSuccess ? 'success' : 'error');
                messageDiv.style.display = 'block';
                setTimeout(() => {
                    messageDiv.style.display = 'none';
                }, 4000);
            }

            function updateIndices() {
                const questionBlocks = questionsContainer.querySelectorAll('.question-block');
                questionBlocks.forEach((qBlock, qIndex) => {
                    qBlock.dataset.index = qIndex;
                    qBlock.querySelector('label').textContent = 'Question ' + (qIndex + 1) + ':';
                    qBlock.querySelector('textarea').name = `questions[${qIndex}][text]`;
                    qBlock.querySelector('input[type="hidden"]').name = `questions[${qIndex}][id]`;

                    const optionsContainer = qBlock.querySelector('.options-container');
                    const optionBlocks = optionsContainer.querySelectorAll('.option-block');
                    optionBlocks.forEach((optBlock, optIndex) => {
                        optBlock.dataset.index = optIndex;
                        optBlock.querySelector('label').textContent = 'Option ' + String.fromCharCode(65 + optIndex) + ':';
                        optBlock.querySelector('input[type="text"]').name = `questions[${qIndex}][options][${optIndex}][text]`;
                        optBlock.querySelector('input[type="checkbox"]').name = `questions[${qIndex}][options][${optIndex}][is_correct]`;
                        optBlock.querySelector('input[type="hidden"]').name = `questions[${qIndex}][options][${optIndex}][id]`;
                    });
                });
            }

            function fetchQuestions() {
                fetch(`manage_exam_questions.php?exam_id=${examId}`)
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            questionsContainer.innerHTML = '';
                            data.questions.forEach((question, qIndex) => {
                                let qHTML = questionTemplate.replace(/__INDEX__/g, qIndex).replace(/__NUMBER__/g, qIndex + 1);
                                questionsContainer.insertAdjacentHTML('beforeend', qHTML);
                                const qBlock = questionsContainer.querySelector(`.question-block[data-index="${qIndex}"]`);
                                qBlock.querySelector('textarea').value = question.question_text;
                                qBlock.querySelector('input[type="hidden"]').value = question.id;

                                const optionsContainer = qBlock.querySelector('.options-container');
                                optionsContainer.innerHTML = '';
                                question.options.forEach((option, optIndex) => {
                                    let optHTML = optionTemplate.replace(/__Q_INDEX__/g, qIndex)
                                        .replace(/__OPT_INDEX__/g, optIndex)
                                        .replace(/__OPT_LETTER__/g, String.fromCharCode(65 + optIndex));
                                    optionsContainer.insertAdjacentHTML('beforeend', optHTML);
                                    const optBlock = optionsContainer.querySelector(`.option-block[data-index="${optIndex}"]`);
                                    optBlock.querySelector('input[type="text"]').value = option.option_text;
                                    optBlock.querySelector('input[type="checkbox"]').checked = option.is_correct == 1;
                                    optBlock.querySelector('input[type="hidden"]').value = option.id;
                                });
                            });
                            updateIndices();
                        } else {
                            showMessage('Failed to load questions: ' + (data.error || 'Unknown error'), false);
                        }
                    })
                    .catch(() => {
                        showMessage('Failed to load questions due to network error.', false);
                    });
            }

            addQuestionBtn.addEventListener('click', () => {
                const newIndex = questionsContainer.querySelectorAll('.question-block').length;
                let newQuestionHTML = questionTemplate.replace(/__INDEX__/g, newIndex).replace(/__NUMBER__/g, newIndex + 1);
                questionsContainer.insertAdjacentHTML('beforeend', newQuestionHTML);
                updateIndices();
            });

            questionsContainer.addEventListener('click', (e) => {
                if (e.target.classList.contains('remove-question-btn')) {
                    e.target.closest('.question-block').remove();
                    updateIndices();
                } else if (e.target.classList.contains('add-option-btn')) {
                    const qBlock = e.target.closest('.question-block');
                    const optionsContainer = qBlock.querySelector('.options-container');
                    const qIndex = qBlock.dataset.index;
                    const newOptIndex = optionsContainer.querySelectorAll('.option-block').length;
                    let newOptionHTML = optionTemplate.replace(/__Q_INDEX__/g, qIndex).replace(/__OPT_INDEX__/g, newOptIndex).replace(/__OPT_LETTER__/g, String.fromCharCode(65 + newOptIndex));
                    optionsContainer.insertAdjacentHTML('beforeend', newOptionHTML);
                    updateIndices();
                } else if (e.target.classList.contains('remove-option-btn')) {
                    e.target.closest('.option-block').remove();
                    updateIndices();
                }
            });

            saveChangesBtn.addEventListener('click', () => {
                const questionBlocks = questionsContainer.querySelectorAll('.question-block');
                const questionsData = [];

                questionBlocks.forEach((qBlock) => {
                    const qIndex = qBlock.dataset.index;
                    const qText = qBlock.querySelector('textarea').value.trim();
                    const qId = parseInt(qBlock.querySelector('input[type="hidden"]').value, 10) || 0;

                    if (qText === '') {
                        return; // skip empty question
                    }

                    const optionsContainer = qBlock.querySelector('.options-container');
                    const optionBlocks = optionsContainer.querySelectorAll('.option-block');
                    const optionsData = [];

                    optionBlocks.forEach((optBlock) => {
                        const optText = optBlock.querySelector('input[type="text"]').value.trim();
                        const optId = parseInt(optBlock.querySelector('input[type="hidden"]').value, 10) || 0;
                        const isCorrect = optBlock.querySelector('input[type="checkbox"]').checked ? 1 : 0;

                        if (optText === '') {
                            return; // skip empty option
                        }

                        optionsData.push({
                            id: optId,
                            text: optText,
                            is_correct: isCorrect
                        });
                    });

                    questionsData.push({
                        id: qId,
                        text: qText,
                        options: optionsData
                    });
                });

                fetch('manage_exam_questions.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        exam_id: examId,
                        questions: questionsData
                    })
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showMessage('Questions and options saved successfully.', true);
                        fetchQuestions();
                    } else {
                        showMessage('Failed to save: ' + (data.error || 'Unknown error'), false);
                    }
                })
                .catch(() => {
                    showMessage('Failed to save due to network error.', false);
                });
            });

            // Initial load
            fetchQuestions();
        });
    </script>
</body>
</html>
